<?php
// api.php - PHP Backend API for Shared Hosting

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Database Configuration
// შეცვალეთ თქვენი cPanel-ის მონაცემებით
$host = 'localhost';
$dbname = 'your_database_name';  // თქვენი database name
$username = 'your_username';      // თქვენი cPanel username
$password = 'your_password';      // თქვენი database password

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database connection failed: ' . $e->getMessage()
    ]);
    exit();
}

// Get request method and action
$method = $_SERVER['REQUEST_METHOD'];
$request = isset($_GET['action']) ? $_GET['action'] : '';

// Route handling
switch($method) {
    case 'GET':
        if ($request === 'bookings') {
            getAllBookings($pdo);
        } elseif ($request === 'booking' && isset($_GET['id'])) {
            getBooking($pdo, $_GET['id']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid request']);
        }
        break;
        
    case 'POST':
        if ($request === 'create') {
            createBooking($pdo);
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid request']);
        }
        break;
        
    case 'PUT':
    case 'PATCH':
        if ($request === 'update' && isset($_GET['id'])) {
            updateBookingStatus($pdo, $_GET['id']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid request']);
        }
        break;
        
    case 'DELETE':
        if ($request === 'delete' && isset($_GET['id'])) {
            deleteBooking($pdo, $_GET['id']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid request']);
        }
        break;
        
    default:
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
        break;
}

// Function: Create new booking
function createBooking($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        echo json_encode(['success' => false, 'message' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required = ['fullName', 'email', 'phone', 'checkIn', 'checkOut', 'roomType', 'guests', 'totalPrice'];
    foreach ($required as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            echo json_encode(['success' => false, 'message' => "Missing field: $field"]);
            return;
        }
    }
    
    // Generate invoice number
    $invoiceNumber = 'INV-' . time() . rand(100, 999);
    
    // Prepare services JSON
    $services = isset($data['services']) ? json_encode($data['services']) : json_encode([]);
    
    try {
        $sql = "INSERT INTO bookings 
                (invoice_number, full_name, email, phone, check_in, check_out, 
                 room_type, guests, services, total_price, booking_date, status)
                VALUES 
                (:invoice_number, :full_name, :email, :phone, :check_in, :check_out,
                 :room_type, :guests, :services, :total_price, NOW(), 'confirmed')";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':invoice_number' => $invoiceNumber,
            ':full_name' => $data['fullName'],
            ':email' => $data['email'],
            ':phone' => $data['phone'],
            ':check_in' => $data['checkIn'],
            ':check_out' => $data['checkOut'],
            ':room_type' => $data['roomType'],
            ':guests' => $data['guests'],
            ':services' => $services,
            ':total_price' => $data['totalPrice']
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'დაჯავშნა წარმატებით შეინახა',
            'invoiceNumber' => $invoiceNumber,
            'bookingId' => $pdo->lastInsertId()
        ]);
        
    } catch(PDOException $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error creating booking: ' . $e->getMessage()
        ]);
    }
}

// Function: Get all bookings
function getAllBookings($pdo) {
    try {
        $sql = "SELECT * FROM bookings ORDER BY booking_date DESC";
        $stmt = $pdo->query($sql);
        $bookings = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'bookings' => $bookings
        ]);
        
    } catch(PDOException $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error fetching bookings: ' . $e->getMessage()
        ]);
    }
}

// Function: Get single booking
function getBooking($pdo, $id) {
    try {
        $sql = "SELECT * FROM bookings WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        $booking = $stmt->fetch();
        
        if ($booking) {
            echo json_encode([
                'success' => true,
                'booking' => $booking
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Booking not found'
            ]);
        }
        
    } catch(PDOException $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error fetching booking: ' . $e->getMessage()
        ]);
    }
}

// Function: Update booking status
function updateBookingStatus($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($data['status'])) {
        echo json_encode(['success' => false, 'message' => 'Status is required']);
        return;
    }
    
    try {
        $sql = "UPDATE bookings SET status = :status WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':status' => $data['status'],
            ':id' => $id
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'სტატუსი წარმატებით განახლდა'
        ]);
        
    } catch(PDOException $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error updating status: ' . $e->getMessage()
        ]);
    }
}

// Function: Delete booking
function deleteBooking($pdo, $id) {
    try {
        $sql = "DELETE FROM bookings WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'დაჯავშნა წარმატებით წაიშალა'
        ]);
        
    } catch(PDOException $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error deleting booking: ' . $e->getMessage()
        ]);
    }
}
?>
